-- Migration: Remove 'disabled' attribute from widget definitions in config.cnft_form
-- Rules:
-- 1. If 'disabled' exists and 'readonly' does not exist -> copy 'disabled' value to 'readonly', then remove 'disabled'
-- 2. If both exist (same or different value) -> keep only 'readonly', remove 'disabled'
-- 3. If only 'readonly' exists -> no change needed

BEGIN;

-- 1) Core function working on jsonb
CREATE OR REPLACE FUNCTION migrate_disabled_to_readonly(input_json jsonb)
  RETURNS jsonb
  LANGUAGE plpgsql
AS $$
DECLARE
  result jsonb;
  key text;
  value jsonb;
  new_widget jsonb;
BEGIN
  IF input_json IS NULL THEN
    RETURN NULL;
  END IF;

  -- ARRAY
  IF jsonb_typeof(input_json) = 'array' THEN
    RETURN (
      SELECT COALESCE(jsonb_agg(migrate_disabled_to_readonly(elem)), '[]'::jsonb)
      FROM jsonb_array_elements(input_json) AS t(elem)
    );
  END IF;

  -- SCALAR
  IF jsonb_typeof(input_json) <> 'object' THEN
    RETURN input_json;
  END IF;

  -- OBJECT
  result := '{}'::jsonb;

  FOR key, value IN SELECT * FROM jsonb_each(input_json) LOOP
      IF key = 'widget' AND jsonb_typeof(value) = 'object' THEN
        new_widget := value;

        IF new_widget ? 'disabled' THEN
          -- copy disabled -> readonly only if readonly missing
          IF NOT (new_widget ? 'readonly') THEN
            new_widget := jsonb_set(new_widget, '{readonly}', new_widget->'disabled', true);
          END IF;

          -- always remove disabled
          new_widget := new_widget - 'disabled';
        END IF;

        result := jsonb_set(result, ARRAY[key], new_widget, true);
      ELSE
        result := jsonb_set(result, ARRAY[key], migrate_disabled_to_readonly(value), true);
      END IF;
    END LOOP;

  RETURN result;
END;
$$;

-- 2) Wrapper for json column (so UPDATE works when schema is type json)
CREATE OR REPLACE FUNCTION migrate_disabled_to_readonly(input_json json)
  RETURNS json
  LANGUAGE sql
AS $$
SELECT migrate_disabled_to_readonly(input_json::jsonb)::json;
$$;

-- Preview changes (uncomment to test)
-- SELECT code,
--        schema as old_schema,
--        migrate_disabled_to_readonly(schema) as new_schema
-- FROM config.cnft_form
-- WHERE code = 'test.proc.tskActTmpl.cod.v1..7738b9';

-- Apply migration to specific row for testing
UPDATE config.cnft_form
SET schema = migrate_disabled_to_readonly(schema);
-- WHERE code = 'test.proc.tskActTmpl.cod.v1..7738b9';

-- After testing, remove the WHERE clause to apply to all rows:
-- UPDATE config.cnft_form
-- SET schema = migrate_disabled_to_readonly(schema);

-- Optional: only touch rows that actually contain "disabled" somewhere
-- UPDATE config.cnft_form
-- SET schema = migrate_disabled_to_readonly(schema)
-- WHERE schema::text LIKE '%"disabled"%';

-- 3) Clean up functions after migration
DROP FUNCTION IF EXISTS migrate_disabled_to_readonly(json);
DROP FUNCTION IF EXISTS migrate_disabled_to_readonly(jsonb);

COMMIT;
-- Migration: Convert 'privilege' attribute to widget 'hidden' and 'readonly' in config.cnft_form
-- Target column: config.cnft_form.schema (json)
--
-- Rules (per property object that has both "privilege" and "widget"):
-- 1) privilege.read  -> widget.hidden   = "${hasPriv('<read>')}" (set/overwrite)
-- 2) privilege.write -> widget.readonly =
--    - if readonly missing: "${hasPriv('<write>')}"
--    - if readonly is boolean true: keep true
--    - if readonly is boolean false: replace with "${hasPriv('<write>')}"
--    - if readonly is a string expr:
--        "${(<existingExpr>) && hasPriv('<write>')}"
--        where <existingExpr> is unwrapped from ${...} when present
-- 3) Remove privilege attribute from the property object
-- 4) Recurse through whole schema

BEGIN;

-- 1) Core function on jsonb
CREATE OR REPLACE FUNCTION migrate_privilege_to_widget(input_json jsonb)
  RETURNS jsonb
  LANGUAGE plpgsql
AS $$
DECLARE
  result jsonb;
  key text;
  value jsonb;

  obj jsonb;
  widget jsonb;
  priv jsonb;
  priv_read text;
  priv_write text;

  existing_readonly jsonb;
  existing_expr text;
  combined_readonly text;
  hidden_expr text;
BEGIN
  IF input_json IS NULL THEN
    RETURN NULL;
  END IF;

  -- ARRAY
  IF jsonb_typeof(input_json) = 'array' THEN
    RETURN (
      SELECT COALESCE(jsonb_agg(migrate_privilege_to_widget(elem)), '[]'::jsonb)
      FROM jsonb_array_elements(input_json) AS t(elem)
    );
  END IF;

  -- SCALAR
  IF jsonb_typeof(input_json) <> 'object' THEN
    RETURN input_json;
  END IF;

  -- OBJECT: if looks like property (has privilege + widget object), transform it
  IF (input_json ? 'privilege')
    AND (input_json ? 'widget')
    AND jsonb_typeof(input_json->'widget') = 'object'
  THEN
    obj := input_json;
    widget := obj->'widget';
    priv := obj->'privilege';

    priv_read := NULL;
    priv_write := NULL;

    IF jsonb_typeof(priv) = 'object' THEN
      priv_read := priv->>'read';
      priv_write := priv->>'write';
    END IF;

    -- privilege.read -> widget.hidden (overwrite / set)
    IF priv_read IS NOT NULL AND priv_read <> '' THEN
      hidden_expr := '${hasPriv(''' || priv_read || ''')}';
      widget := jsonb_set(widget, '{hidden}', to_jsonb(hidden_expr), true);
      -- Pokud bys chtěl "skrýt když NEMÁ", tak místo toho:
      -- hidden_expr := '${!hasPriv(''' || priv_read || ''')}';
    END IF;

    -- privilege.write -> widget.readonly
    IF priv_write IS NOT NULL AND priv_write <> '' THEN
      existing_readonly := widget->'readonly';

      IF existing_readonly IS NULL THEN
        combined_readonly := '${hasPriv(''' || priv_write || ''')}';
        widget := jsonb_set(widget, '{readonly}', to_jsonb(combined_readonly), true);

      ELSIF jsonb_typeof(existing_readonly) = 'boolean' THEN
        -- keep true, replace false
        IF existing_readonly = 'false'::jsonb THEN
          combined_readonly := '${hasPriv(''' || priv_write || ''')}';
          widget := jsonb_set(widget, '{readonly}', to_jsonb(combined_readonly), true);
        END IF;

      ELSIF jsonb_typeof(existing_readonly) = 'string' THEN
        existing_expr := existing_readonly::text;         -- includes quotes
        existing_expr := trim(both '"' from existing_expr); -- remove quotes

        -- unwrap ${...} when present
        IF left(existing_expr, 2) = '${' AND right(existing_expr, 1) = '}' THEN
          existing_expr := substring(existing_expr from 3 for char_length(existing_expr) - 3);
        END IF;

        combined_readonly := '${(' || existing_expr || ') && hasPriv(''' || priv_write || ''')}';
        widget := jsonb_set(widget, '{readonly}', to_jsonb(combined_readonly), true);

      ELSE
        -- readonly exists but is not boolean/string (object/array/number) -> leave as-is (safer)
        NULL;
      END IF;
    END IF;

    -- write back widget + drop privilege
    obj := jsonb_set(obj, '{widget}', widget, true);
    obj := obj - 'privilege';

    -- recurse into the rest (keep widget as final, don't recurse into it)
    result := '{}'::jsonb;
    FOR key, value IN SELECT * FROM jsonb_each(obj) LOOP
        IF key = 'widget' THEN
          result := jsonb_set(result, ARRAY[key], value, true);
        ELSE
          result := jsonb_set(result, ARRAY[key], migrate_privilege_to_widget(value), true);
        END IF;
      END LOOP;

    RETURN result;
  END IF;

  -- Default: recurse into all fields
  result := '{}'::jsonb;
  FOR key, value IN SELECT * FROM jsonb_each(input_json) LOOP
      result := jsonb_set(result, ARRAY[key], migrate_privilege_to_widget(value), true);
    END LOOP;

  RETURN result;
END;
$$;

-- 2) Wrapper for json column
CREATE OR REPLACE FUNCTION migrate_privilege_to_widget(input_json json)
  RETURNS json
  LANGUAGE sql
AS $$
SELECT migrate_privilege_to_widget(input_json::jsonb)::json;
$$;

-- Preview (doporučuju)
-- SELECT code,
--        schema as old_schema,
--        migrate_privilege_to_widget(schema) as new_schema
-- FROM config.cnft_form
-- WHERE code = 'neco';

-- Apply migration to specific row for testing
UPDATE config.cnft_form
SET schema = migrate_privilege_to_widget(schema);
-- WHERE code = 'test.proc.tskActTmpl.cod.v1..7738b9';

-- Apply to all rows that contain privilege (volitelně)
-- UPDATE config.cnft_form
-- SET schema = migrate_privilege_to_widget(schema)
-- WHERE schema::text LIKE '%"privilege"%';

-- 3) Clean up
DROP FUNCTION IF EXISTS migrate_privilege_to_widget(json);
DROP FUNCTION IF EXISTS migrate_privilege_to_widget(jsonb);

COMMIT;
-- Migration: Update tsmControls id values in config.cnft_form.schema
-- This migration transforms old control IDs to new namespaced format
-- Example: "severity" -> "Ticket.New.Severity"
-- Uses entity_type column to determine correct mapping for ambiguous fields

BEGIN;

-- Create a temporary mapping table for ID transformations
-- entity_type NULL means the mapping applies to all entity types (unique keys)
CREATE TEMP TABLE id_mapping (
    entity_type TEXT,
    old_id TEXT,
    new_id TEXT NOT NULL,
    PRIMARY KEY (entity_type, old_id)
);

-- ============================================================================
-- Address (entityType: Address)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Address', 'formattedAddress', 'Address.New.FormattedAddress'),
('Address', 'postOffice', 'Address.New.PostOffice'),
('Address', 'country', 'Address.New.Country'),
('Address', 'region', 'Address.New.Region'),
('Address', 'district', 'Address.New.District'),
('Address', 'city', 'Address.New.City'),
('Address', 'partOfCity', 'Address.New.PartOfCity'),
('Address', 'street', 'Address.New.Street'),
('Address', 'externalId', 'Address.New.ExternalId'),
('Address', 'propertyNumber', 'Address.New.PropertyNumber'),
('Address', 'regId', 'Address.New.RegId'),
('Address', 'verified', 'Address.New.Verified'),
('Address', 'verificationDate', 'Address.New.VerificationDate'),
('Address', 'replacedById', 'Address.New.ReplacedById'),
('Address', 'rejectedConflicts', 'Address.New.RejectedConflicts'),
('Address', 'streetNumberNo', 'Address.New.StreetNumberNo'),
('Address', 'streetNumberLet', 'Address.New.StreetNumberLet'),
('Address', 'lat', 'Address.New.Lat'),
('Address', 'lon', 'Address.New.Lon'),
('Address', 'description', 'Address.New.Description');

-- ============================================================================
-- BillingDocument (entityType: BillingDocument)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('BillingDocument', 'name', 'BillingDocument.New.Name'),
('BillingDocument', 'invoiceDate', 'BillingDocument.New.InvoiceDate'),
('BillingDocument', 'dueDate', 'BillingDocument.New.DueDate'),
('BillingDocument', 'dateOfRealization', 'BillingDocument.New.DateOfRealization'),
('BillingDocument', 'variableSymbol', 'BillingDocument.New.VariableSymbol'),
('BillingDocument', 'customerId', 'BillingDocument.New.CustomerId'),
('BillingDocument', 'accountId', 'BillingDocument.New.AccountId'),
('BillingDocument', 'invoiceNumber', 'BillingDocument.New.InvoiceNumber'),
('BillingDocument', 'billingCurrency', 'BillingDocument.New.BillingCurrency'),
('BillingDocument', 'billingCycle', 'BillingDocument.New.BillingCycle'),
('BillingDocument', 'status', 'BillingDocument.New.Status'),
('BillingDocument', 'description', 'BillingDocument.New.Description');

-- ============================================================================
-- BillingDocumentLine (entityType: BillingDocumentLine)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('BillingDocumentLine', 'name', 'BillingDocumentLine.New.Name'),
('BillingDocumentLine', 'parentDocumentLineId', 'BillingDocumentLine.New.ParentDocumentLineId'),
('BillingDocumentLine', 'billingDocumentId', 'BillingDocumentLine.New.BillingDocumentId'),
('BillingDocumentLine', 'entityInstanceConfigurationId', 'BillingDocumentLine.New.EntityInstanceConfigurationId'),
('BillingDocumentLine', 'vatRate', 'BillingDocumentLine.New.VatRate'),
('BillingDocumentLine', 'priceWithoutVat', 'BillingDocumentLine.New.PriceWithoutVat'),
('BillingDocumentLine', 'sortOrder', 'BillingDocumentLine.New.SortOrder');

-- ============================================================================
-- EntityCatalogCategory (entityType: EntityCatalogCategory)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('EntityCatalogCategory', 'categoryName', 'EntityCatalogCategory.New.CategoryName'),
('EntityCatalogCategory', 'categoryCode', 'EntityCatalogCategory.New.CategoryCode'),
('EntityCatalogCategory', 'entityCatalog', 'EntityCatalogCategory.New.EntityCatalog'),
('EntityCatalogCategory', 'parent', 'EntityCatalogCategory.New.Parent'),
('EntityCatalogCategory', 'entitySpecificationSpecId', 'EntityCatalogCategory.New.EntitySpecificationSpecId'),
('EntityCatalogCategory', 'icon', 'EntityCatalogCategory.New.Icon'),
('EntityCatalogCategory', 'sortOrder', 'EntityCatalogCategory.New.SortOrder'),
('EntityCatalogCategory', 'description', 'EntityCatalogCategory.New.Description');

-- ============================================================================
-- EntityCatalogSpecification (entityType: EntityCatalogSpecification)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('EntityCatalogSpecification', 'specificationName', 'EntityCatalogSpecification.New.SpecificationName'),
('EntityCatalogSpecification', 'specificationCode', 'EntityCatalogSpecification.New.SpecificationCode'),
('EntityCatalogSpecification', 'catalogCategory', 'EntityCatalogSpecification.New.CatalogCategory'),
('EntityCatalogSpecification', 'entityInstanceSpecId', 'EntityCatalogSpecification.New.EntityInstanceSpecId'),
('EntityCatalogSpecification', 'orderLineSpecId', 'EntityCatalogSpecification.New.OrderLineSpecId'),
('EntityCatalogSpecification', 'icon', 'EntityCatalogSpecification.New.Icon'),
('EntityCatalogSpecification', 'sortOrder', 'EntityCatalogSpecification.New.SortOrder'),
('EntityCatalogSpecification', 'description', 'EntityCatalogSpecification.New.Description'),
('EntityCatalogSpecification', 'instantiable', 'EntityCatalogSpecification.New.Instantiable'),
('EntityCatalogSpecification', 'lifecycleStatus', 'EntityCatalogSpecification.New.LifecycleStatus');

-- ============================================================================
-- RegisterValue (entityType: RegisterValue)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('RegisterValue', 'name', 'RegisterValue.New.Name'),
('RegisterValue', 'code', 'RegisterValue.New.Code'),
('RegisterValue', 'order', 'RegisterValue.New.Order'),
('RegisterValue', 'parent', 'RegisterValue.New.Parent'),
('RegisterValue', 'icon', 'RegisterValue.New.Icon'),
('RegisterValue', 'validFrom', 'RegisterValue.New.ValidFrom'),
('RegisterValue', 'validTo', 'RegisterValue.New.ValidTo'),
('RegisterValue', 'description', 'RegisterValue.New.Description');

-- ============================================================================
-- Customer (entityType: Customer)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Customer', 'customerName', 'Customer.New.Name'),
('Customer', 'legalForm', 'Customer.New.LegalForm'),
('Customer', 'customerSegment', 'Customer.New.Segment'),
('Customer', 'persIdentNum', 'Customer.New.IdentificationNumber'),
('Customer', 'taxIdentNum', 'Customer.New.Vatin'),
('Customer', 'description', 'Customer.New.Description');

-- ============================================================================
-- Comment (entityType: Comment)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Comment', 'comment', 'Comment.New.Comment');

-- ============================================================================
-- EntityInstanceConfiguration (entityType: EntityInstanceConfiguration)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('EntityInstanceConfiguration', 'name', 'EntityInstanceConfiguration.New.Name'),
('EntityInstanceConfiguration', 'key', 'EntityInstanceConfiguration.New.Key'),
('EntityInstanceConfiguration', 'customerId', 'EntityInstanceConfiguration.New.CustomerId'),
('EntityInstanceConfiguration', 'accountId', 'EntityInstanceConfiguration.New.AccountId'),
('EntityInstanceConfiguration', 'validFrom', 'EntityInstanceConfiguration.New.ValidFrom'),
('EntityInstanceConfiguration', 'validTo', 'EntityInstanceConfiguration.New.ValidTo'),
('EntityInstanceConfiguration', 'description', 'EntityInstanceConfiguration.New.Description'),
('EntityInstanceConfiguration', 'location1', 'EntityInstanceConfiguration.New.Location1'),
('EntityInstanceConfiguration', 'location2', 'EntityInstanceConfiguration.New.Location2');

-- ============================================================================
-- Order (entityType: Order)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Order', 'key', 'Order.New.Key'),
('Order', 'customerId', 'Order.New.CustomerId'),
('Order', 'accountId', 'Order.New.AccountId'),
('Order', 'entityInstanceConfiguration', 'Order.New.EntityInstanceConfiguration'),
('Order', 'deliveryDateRequested', 'Order.New.DeliveryDateRequested'),
('Order', 'subject', 'Order.New.Subject'),
('Order', 'description', 'Order.New.Description');

-- ============================================================================
-- Stock (entityType: Stock)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Stock', 'name', 'Stock.New.Name'),
('Stock', 'ownerUserId', 'Stock.New.OwnerUserId'),
('Stock', 'def', 'Stock.New.Def'),
('Stock', 'status', 'Stock.New.Status'),
('Stock', 'description', 'Stock.New.Description');

-- ============================================================================
-- Ticket (entityType: Ticket)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Ticket', 'name', 'Ticket.New.Name'),
('Ticket', 'severity', 'Ticket.New.Severity'),
('Ticket', 'priority', 'Ticket.New.Priority'),
('Ticket', 'channel', 'Ticket.New.Channel'),
('Ticket', 'ticketCategory', 'Ticket.New.TicketCategory'),
('Ticket', 'externalId', 'Ticket.New.ExternalId'),
('Ticket', 'creationGroup', 'Ticket.New.CreationGroup'),
('Ticket', 'description', 'Ticket.New.Description');


-- Create a function to recursively update IDs in the layout structure
-- Takes entity_type as parameter for context-aware mapping
CREATE OR REPLACE FUNCTION pg_temp.migrate_layout_ids(layout_item JSONB, p_entity_type TEXT)
RETURNS JSONB AS $
DECLARE
    result JSONB;
    new_id TEXT;
    items_array JSONB;
    updated_items JSONB;
    i INT;
BEGIN
    IF layout_item IS NULL THEN
        RETURN NULL;
    END IF;

    result := layout_item;

    -- If this item has an 'id' field, check if it needs to be migrated
    IF result ? 'id' AND jsonb_typeof(result -> 'id') = 'string' THEN
        -- First try entity-specific mapping
        SELECT m.new_id INTO new_id
        FROM id_mapping m
        WHERE m.entity_type = p_entity_type
          AND m.old_id = result ->> 'id';

        IF new_id IS NOT NULL THEN
            result := jsonb_set(result, '{id}', to_jsonb(new_id));
        END IF;
    END IF;

    -- If this item has nested 'items' array, recursively process them
    IF result ? 'items' AND jsonb_typeof(result -> 'items') = 'array' THEN
        updated_items := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'items') - 1 LOOP
            updated_items := updated_items || jsonb_build_array(
                pg_temp.migrate_layout_ids(result -> 'items' -> i, p_entity_type)
            );
        END LOOP;
        result := jsonb_set(result, '{items}', updated_items);
    END IF;

    RETURN result;
END;
$ LANGUAGE plpgsql;

-- Create a function to process the entire schema
CREATE OR REPLACE FUNCTION pg_temp.migrate_schema(schema_data JSONB, p_entity_type TEXT)
RETURNS JSONB AS $
DECLARE
    result JSONB;
    layout_array JSONB;
    updated_layout JSONB;
    i INT;
BEGIN
    IF schema_data IS NULL THEN
        RETURN NULL;
    END IF;

    result := schema_data;

    -- Process the 'layout' array if it exists
    IF result ? 'layout' AND jsonb_typeof(result -> 'layout') = 'array' THEN
        updated_layout := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'layout') - 1 LOOP
            updated_layout := updated_layout || jsonb_build_array(
                pg_temp.migrate_layout_ids(result -> 'layout' -> i, p_entity_type)
            );
        END LOOP;
        result := jsonb_set(result, '{layout}', updated_layout);
    END IF;

    RETURN result;
END;
$ LANGUAGE plpgsql;

-- ============================================================================
-- Preview changes (uncomment to test before applying)
-- ============================================================================
-- SELECT
--     id,
--     entity_type,
--     schema AS old_schema,
--     pg_temp.migrate_schema(schema::jsonb, entity_type) AS new_schema
-- FROM config.cnft_form
-- WHERE schema IS NOT NULL
--   AND schema::jsonb ? 'layout'
--   AND entity_type IS NOT NULL
--   -- AND id = 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'  -- Uncomment to test on specific form
-- LIMIT 10;

-- ============================================================================
-- Apply the migration
-- ============================================================================
UPDATE config.cnft_form
SET schema = pg_temp.migrate_schema(schema::jsonb, entity_type)::json
WHERE schema IS NOT NULL
  AND schema::jsonb ? 'layout'
  AND entity_type IS NOT NULL
  -- AND id = 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'  -- Uncomment to test on specific form
;

-- Report how many rows were updated
DO $
DECLARE
    updated_count INT;
BEGIN
    GET DIAGNOSTICS updated_count = ROW_COUNT;
    RAISE NOTICE 'Updated % rows in config.cnft_form', updated_count;
END $;

COMMIT;
-- Migration: Wrap Ticket.Accordions.Description control in its own dtl-fluent-inplace
-- This migration finds the Ticket.Accordions.Description control and wraps it
-- in a separate dtl-fluent-inplace layout, updating accordion content references
-- Handles both cases:
--   1. Description is directly in accordion items (not wrapped)
--   2. Description is inside a dtl-fluent-inplace wrapper (already wrapped with other items)

BEGIN;

-- Recursive function to process the layout structure
CREATE OR REPLACE FUNCTION pg_temp.process_layout_recursive(item JSONB)
RETURNS JSONB AS $func$
DECLARE
    result JSONB;
    items_array JSONB;
    new_items JSONB;
    current_item JSONB;
    processed_item JSONB;
    i INT;
    j INT;
    description_item JSONB;
    description_index INT;
    found_description BOOLEAN;
    found_in_inplace BOOLEAN;
    inner_items JSONB;
    new_inner_items JSONB;
    accordion_config JSONB;
    new_accordions JSONB;
    accordion JSONB;
    content_array JSONB;
    new_content JSONB;
BEGIN
    IF item IS NULL THEN
        RETURN NULL;
    END IF;

    result := item;

    -- Check if this is the dtl-fluent-accordion container we're looking for
    IF result ? 'widget'
       AND result -> 'widget' ->> 'type' = 'dtl-fluent-accordion'
       AND result ? 'items'
       AND jsonb_typeof(result -> 'items') = 'array' THEN

        items_array := result -> 'items';
        new_items := '[]'::JSONB;
        found_description := FALSE;
        found_in_inplace := FALSE;
        description_item := NULL;
        description_index := -1;

        -- First pass: find the Description control
        FOR i IN 0..jsonb_array_length(items_array) - 1 LOOP
            current_item := items_array -> i;

            -- Case 1: Check if Description is directly in accordion items
            IF current_item ->> 'id' = 'Ticket.Accordions.Description' THEN
                description_item := current_item;
                found_description := TRUE;
                found_in_inplace := FALSE;
                description_index := i;
                -- Don't add to new_items yet, we'll wrap it later
                CONTINUE;
            END IF;

            -- Case 2: Check if this is a dtl-fluent-inplace that contains Ticket.Accordions.Description
            IF current_item ? 'widget'
               AND current_item -> 'widget' ->> 'type' = 'dtl-fluent-inplace'
               AND current_item ? 'items'
               AND jsonb_typeof(current_item -> 'items') = 'array' THEN

                inner_items := current_item -> 'items';
                new_inner_items := '[]'::JSONB;

                FOR j IN 0..jsonb_array_length(inner_items) - 1 LOOP
                    IF inner_items -> j ->> 'id' = 'Ticket.Accordions.Description' THEN
                        -- Found the Description control inside inplace, save it for wrapping
                        description_item := inner_items -> j;
                        found_description := TRUE;
                        found_in_inplace := TRUE;
                        description_index := i;
                    ELSE
                        -- Keep other items in the original wrapper
                        new_inner_items := new_inner_items || jsonb_build_array(inner_items -> j);
                    END IF;
                END LOOP;

                -- Add the modified wrapper (now without Description)
                current_item := jsonb_set(current_item, '{items}', new_inner_items);
            END IF;

            new_items := new_items || jsonb_build_array(current_item);
        END LOOP;

        -- If we found the Description, add a new wrapper for it
        IF found_description AND description_item IS NOT NULL THEN
            -- Create new dtl-fluent-inplace wrapper for Description
            new_items := new_items || jsonb_build_array(
                jsonb_build_object(
                    'type', 'layout',
                    'widget', jsonb_build_object('type', 'dtl-fluent-inplace'),
                    'items', jsonb_build_array(description_item)
                )
            );

            result := jsonb_set(result, '{items}', new_items);

            -- Update accordion content references
            -- The new Description wrapper is now at the end of items array
            IF result ? 'config' AND result -> 'config' ? 'accordions' THEN
                accordion_config := result -> 'config' -> 'accordions';
                new_accordions := '[]'::JSONB;

                FOR i IN 0..jsonb_array_length(accordion_config) - 1 LOOP
                    accordion := accordion_config -> i;

                    -- Check if this accordion references the old Description index
                    IF accordion ? 'content' AND jsonb_typeof(accordion -> 'content') = 'array' THEN
                        content_array := accordion -> 'content';
                        new_content := '[]'::JSONB;

                        FOR j IN 0..jsonb_array_length(content_array) - 1 LOOP
                            -- If this references the old index that had Description
                            IF (content_array -> j)::INT = description_index THEN
                                IF found_in_inplace THEN
                                    -- Was in inplace: add both the new wrapper index and the old (now empty) wrapper
                                    new_content := jsonb_build_array(jsonb_array_length(new_items) - 1) || new_content;
                                    new_content := new_content || jsonb_build_array(content_array -> j);
                                ELSE
                                    -- Was direct: just replace with new wrapper index
                                    new_content := new_content || jsonb_build_array(jsonb_array_length(new_items) - 1);
                                END IF;
                            ELSE
                                new_content := new_content || jsonb_build_array(content_array -> j);
                            END IF;
                        END LOOP;

                        accordion := jsonb_set(accordion, '{content}', new_content);
                    END IF;

                    new_accordions := new_accordions || jsonb_build_array(accordion);
                END LOOP;

                result := jsonb_set(result, '{config,accordions}', new_accordions);
            END IF;

            RETURN result;
        ELSE
            -- No changes needed at this level, but still process children
            result := jsonb_set(result, '{items}', new_items);
        END IF;

        RETURN result;
    END IF;

    -- Recursively process 'items' array if present
    IF result ? 'items' AND jsonb_typeof(result -> 'items') = 'array' THEN
        new_items := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'items') - 1 LOOP
            processed_item := pg_temp.process_layout_recursive(result -> 'items' -> i);
            new_items := new_items || jsonb_build_array(processed_item);
        END LOOP;
        result := jsonb_set(result, '{items}', new_items);
    END IF;

    -- Recursively process 'layout' array if present
    IF result ? 'layout' AND jsonb_typeof(result -> 'layout') = 'array' THEN
        new_items := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'layout') - 1 LOOP
            processed_item := pg_temp.process_layout_recursive(result -> 'layout' -> i);
            new_items := new_items || jsonb_build_array(processed_item);
        END LOOP;
        result := jsonb_set(result, '{layout}', new_items);
    END IF;

    RETURN result;
END;
$func$ LANGUAGE plpgsql;

-- ============================================================================
-- Preview changes (uncomment to test before applying)
-- ============================================================================
-- SELECT
--     id,
--     entity_type,
--     schema AS old_schema,
--     pg_temp.process_layout_recursive(schema::jsonb) AS new_schema
-- FROM config.cnft_form
-- WHERE schema IS NOT NULL
--   AND schema::text LIKE '%Ticket.Accordions.Description%'
-- LIMIT 5;

-- ============================================================================
-- Apply the migration
-- ============================================================================
UPDATE config.cnft_form
SET schema = pg_temp.process_layout_recursive(schema::jsonb)::json
WHERE schema IS NOT NULL
  AND schema::text LIKE '%Ticket.Accordions.Description%';

-- Report how many rows were updated
DO $do$
DECLARE
    updated_count INT;
BEGIN
    GET DIAGNOSTICS updated_count = ROW_COUNT;
    RAISE NOTICE 'Updated % rows in config.cnft_form for Ticket.Accordions.Description wrapping', updated_count;
END $do$;

COMMIT;
-- Migration: Replace $value with $context.form in detail forms
-- This migration applies only to rows where baseForm ends with '-detail' or '-new'
-- Replaces all occurrences of "$value" with "$context.form" in the schema column

BEGIN;

-- ============================================================================
-- Preview changes (uncomment to test before applying)
-- ============================================================================
-- SELECT
--     id,
--     base_form,
--     schema AS old_schema,
--     replace(schema::text, '$value', '$context.form')::json AS new_schema
-- FROM config.cnft_form
-- WHERE schema IS NOT NULL
--   AND (base_form LIKE '%-detail' OR base_form LIKE '%-new')
--   AND schema::text LIKE '%$value%'
-- LIMIT 10;

-- ============================================================================
-- Apply the migration
-- ============================================================================
UPDATE config.cnft_form
SET schema = replace(schema::text, '$value', '$context.form')::json
WHERE schema IS NOT NULL
  AND (base_form LIKE '%-detail' OR base_form LIKE '%-new')
  AND schema::text LIKE '%$value%';

-- Report how many rows were updated
DO $do$
DECLARE
    updated_count INT;
BEGIN
    GET DIAGNOSTICS updated_count = ROW_COUNT;
    RAISE NOTICE 'Updated % rows in config.cnft_form for $value -> $context.form migration', updated_count;
END $do$;

COMMIT;
